#######################################################################
# R code: Exercise 11.6 (LSTAR-CNF model forecasts)
# File: LSTAR-CNF-forecast.r
# Coded by: Dao Li
#
# Reference: 
# Li, D. and He, C. (2013).
#    Forecasting with vector nonlinear time series models.
#    Working papers 2013:8, Dalarna University, Sweden.
#    Available at:
#    http://www.diva-portal.org/smash/get/diva2:606647/FULLTEXT02.pdf.
#######################################################################
library("MASS")
rm(list=ls())
myestimation <- function(data,p=3,dn=1,d=7,rank=1){
# Notes: p = 3 (order of LSTAR model); 
#        d = 7 (delay of transition variable)
  n <- length(data[,1])
  T <- length(data[1,])
  y <- data
  tempx <- matrix(0,(n*p),(T-max(p,d)))
  for(i in 1:p){
    tempx[(1+n*(i-1)):(n+n*(i-1)),] <- y[,(max(p,d)-i+1):(T-i)]
  }
  tempy <- y[,(max(p,d)+1):T]
  x     <- rbind(rep(1,(T-max(p,d))),tempx)
  st    <- y[dn,max(p+1-d,1):(T-d)]
  st1   <- round(quantile(st,0.05),2)
  st2   <- round(quantile(st,0.95),2)
# Grid search of (gamma,c)       
  r     <- seq(30,60.4,0.01)   
  c     <- seq(0.69,0.75,0.01) 
  len1  <- length(r)
  len2  <- length(c)
  estl  <- matrix(0,len1,len2)
  for(k in 1:len1){     # search for gamma
      for(j in 1:len2){ # search for c
          ###############################
          # Estimation given one (gamma,c)
          z <- matrix(0,(n*p+1),(T-max(p,d)))
          for(i in 1:(n*p+1)){
            z[i,] <- x[i,]/(1+exp(-r[k]*(st-c[j])/sd(st))) 
          }       
      X      <- rbind(x,z)
      resv   <- tempy - tempy%*%t(X)%*%solve(X%*%t(X))%*%X
      Sigv   <- resv%*%t(resv)/T 
      Sigyz  <- tempy%*%t(z)/T
      Sigyx  <- tempy%*%t(x)/T
      Sigxx  <- x%*%t(x)/T
      Sigxz  <- x%*%t(z)/T
      Sigzz  <- z%*%t(z)/T
      Sigyzx <- Sigyz - Sigyx%*%solve(Sigxx)%*%Sigxz
      Sigzzx <- Sigzz - t(Sigxz)%*%solve(Sigxx)%*%Sigxz

      tempres      <- eigen(x=Sigv,symmetric=TRUE,only.values=FALSE)
Sigv2        <- tempres$vector%*%diag(sqrt(tempres$values))%*%t(tempres$vector)
matrix_eigen <- solve(Sigv2)%*%Sigyzx%*%solve(Sigzzx)%*%t(Sigyzx)%*%solve(Sigv2)
result_eigen <- eigen(x=matrix_eigen,symmetric=TRUE,only.values=FALSE)
      eigen_value  <- result_eigen$values
      eigen_vec    <- result_eigen$vectors
      eigenvec     <- matrix(0,n,rank)
      for(ind in 1:rank){
        eigenvec[,ind] <- eigen_vec[,ind]#each clmn is an eigen vector
        eigenvec[,ind] <- eigenvec[,ind]/sqrt(sum(eigenvec[,ind]^2))   
      }
    A     <- Sigv2%*%eigenvec # n times rank matrix
    B     <- t(eigenvec)%*%solve(Sigv2)%*%Sigyzx%*%solve(Sigzzx)
    Gamma <- A%*%B
    Phi   <- (Sigyx - A%*%B%*%t(Sigxz))%*%solve(Sigxx)
        
    reduced_v <- tempy - Phi%*%x - Gamma%*%z
    normA <- solve(Sigv2)%*%reduced_v%*%t(reduced_v)%*%solve(Sigv2)/T
    estl[k,j] <- sum(diag(normA))
    } # end search for c
  }   # end search for gamma
  inrowcol <- which(estl==min(estl),arr.ind=TRUE)
  estr <- r[inrowcol[1,1]]
  estc <- c[inrowcol[1,2]]
  z    <- matrix(0,(n*p+1),(T-max(p,d)))
  for(i in 1:(n*p+1)){
      z[i,] <- x[i,]/(1+exp(-estr*(st-estc)/sd(st))) 
  }       
  X     <- rbind(x,z)
  resv  <- tempy - tempy%*%t(X)%*%solve(X%*%t(X),tol=1e-30)%*%X    
  Sigv  <- resv%*%t(resv)/T 
  Sigyz <- tempy%*%t(z)/T
  Sigyx <- tempy%*%t(x)/T
  Sigxx <- x%*%t(x)/T
  Sigxz <- x%*%t(z)/T
  Sigzz <- z%*%t(z)/T
                        
  Sigyzx <- Sigyz - Sigyx%*%solve(Sigxx)%*%Sigxz
  Sigzzx <- Sigzz - t(Sigxz)%*%solve(Sigxx)%*%Sigxz

  tempres <- eigen(x=Sigv,symmetric=TRUE,only.values=FALSE)
  Sigv2        <- tempres$vector%*%diag(sqrt(tempres$values))%*%t(tempres$vector)
  matrix_eigen <- solve(Sigv2)%*%Sigyzx%*%solve(Sigzzx)%*%t(Sigyzx)%*%solve(Sigv2)
  result_eigen <- eigen(x=matrix_eigen,symmetric=TRUE,only.values=FALSE)
  eigen_value  <- result_eigen$values
  eigen_vec    <- result_eigen$vectors
  eigenvec     <- matrix(0,n,rank)
  for(ind in 1:rank){
     eigenvec[,ind] <- eigen_vec[,ind]/sqrt(sum(eigen_vec[,ind]^2)) 
   }
        
   A         <- Sigv2%*%eigenvec #n times rank matrix
   B         <- t(eigenvec)%*%solve(Sigv2)%*%Sigyzx%*%solve(Sigzzx)
   Gamma     <- A%*%B
   Phi       <- (Sigyx - A%*%B%*%t(Sigxz))%*%solve(Sigxx)
   reduced_v <- tempy - Phi%*%x - Gamma%*%z
return(list(gamma=estr,c=estc,estPhi=Phi,estA=A,estB=B,residual=reduced_v))
}

myoneT2forecast <- function(y,result,resB,h=12,j,Nb=1000) 
# Notes: Nb = Number of bootstrap replicates; 
#        h  = Maximum forecast horizon
#        2 variables, delay transition variable = 7
# Do one-step ahead forecast up to h here
{
 oneT2newy <- matrix(0,2,h)
 Tused     <- length(y[1,])
 r         <- result$gamma
 c         <- result$c
 Phi       <- result$estPhi
 A         <- result$estA
 B         <- result$estB
 vhatm     <- matrix(rowMeans(resB),2,1)     
 vhat      <- resB - matrix(1,1,Tused-7)%x%vhatm
 Cfactor   <- t(chol(cov(t(vhat))))
 vhat_ind  <- solve(Cfactor)%*%vhat
 AB        <- A%*%B
 sdst      <- sd(y[1,1:(Tused-7)])
 ###################### h=1
 if(h==1){
   tempG <- 1/(1+exp(-r*(y[1,(Tused-7+1)]-c)/sdst))
   oneT2newy[,h] <- Phi[,1]+(Phi[,2:3]+AB[,2:3]*tempG)%*%y[,(Tused)]
                    +(Phi[,4:5]+AB[,4:5]*tempG)%*%y[,(Tused-1)]
                    +(Phi[,6:7]+AB[,6:7]*tempG)%*%y[,(Tused-2)]
    }
 ###################### h>1
 if(h>1 && h<=j){
    temp1     <- sample(vhat_ind[1,],size=Nb*h,replace=TRUE)
    temp2     <- sample(vhat_ind[2,],size=Nb*h,replace=TRUE)
    vboots    <- rbind(temp1,temp2)
    vboots    <- Cfactor%*%vboots
    tempvhat1 <- matrix(vboots[1,]+rep(vhatm[1,],Nb*h),nrow=Nb,ncol=h)
    tempvhat2 <- matrix(vboots[2,]+rep(vhatm[2,],Nb*h),nrow=Nb,ncol=h)
        
    tempy1    <- matrix(y[,(Tused)],2,Nb)#each y(t-1) at Tused~Tused-6
    tempy2    <- matrix(y[,(Tused-1)],2,Nb)
    tempy3    <- matrix(y[,(Tused-2)],2,Nb)
    tempy     <- matrix(0,2,Nb)
    tempst    <- rep(0,7)
    for(i in 1:h){       
     if(i<=7){
     for(k in 1:Nb){
       tempvhat  <- rbind(tempvhat1[k,i],tempvhat2[k,i])
       tempG     <- 1/(1+exp(-r*(y[1,(Tused-7+i)]-c)/sdst))#st=y1(t-7)
       tempy[,k] <- Phi[,1]+(Phi[,2:3]+AB[,2:3]*tempG)%*%tempy1[,k]
                    +(Phi[,4:5]+AB[,4:5]*tempG)%*%tempy2[,k]
                    +(Phi[,6:7]+AB[,6:7]*tempG)%*%tempy3[,k]+tempvhat 
        } # end loop for k            
       tempy3    <- tempy2
       tempy2    <- tempy1
       tempy1    <- tempy 
       tempst[i] <- mean(tempy[1,])                
      }  # end loop for i<7
      if(i>7){
        for(k in 1:Nb){
          tempvhat  <- rbind(tempvhat1[k,i],tempvhat2[k,i])
          tempG     <- 1/(1+exp(-r*(tempst[i-7]-c)/sdst))#st=y1(t-7)
          tempy[,k] <- Phi[,1]+(Phi[,2:3]+AB[,2:3]*tempG)%*%tempy1[,k]
                       +(Phi[,4:5]+AB[,4:5]*tempG)%*%tempy2[,k]
                       +(Phi[,6:7]+AB[,6:7]*tempG)%*%tempy3[,k]+tempvhat
         }  # end loop for k             
       tempy3   <- tempy2
       tempy2   <- tempy1
       tempy1   <- tempy 
       }  # end loop for i>7 
       oneT2newy[,i] <- matrix(c(mean(tempy[1,]),mean(tempy[2,])),2,1)
     } # end loop for i in 1:h
  }    # end loop for h>1 and h<=j
if(h>1 && h>j){
   temp1     <- sample(vhat_ind[1,],size=Nb*h,replace=TRUE)
   temp2     <- sample(vhat_ind[2,],size=Nb*h,replace=TRUE)
   vboots    <- rbind(temp1,temp2)
   vboots    <- Cfactor%*%vboots
   tempvhat1 <- matrix(vboots[1,]+rep(vhatm[1,],Nb*h),nrow=Nb,ncol=h)
   tempvhat2 <- matrix(vboots[2,]+rep(vhatm[2,],Nb*h),nrow=Nb,ncol=h)
        
   tempy1    <- matrix(y[,(Tused)],2,Nb)#each y(t-1) at Tused~Tused-6
   tempy2    <- matrix(y[,(Tused-1)],2,Nb)
   tempy3    <- matrix(y[,(Tused-2)],2,Nb)
   tempy     <- matrix(0,2,Nb)
   tempst    <- rep(0,7)
   for(i in 1:j){       
     if(i<=7){
      for(k in 1:Nb) {
        tempvhat  <- rbind(tempvhat1[k,i],tempvhat2[k,i])
        tempG     <- 1/(1+exp(-r*(y[1,(Tused-7+i)]-c)/sdst))#st=y1(t-7)
        tempy[,k] <- Phi[,1]+(Phi[,2:3]+AB[,2:3]*tempG)%*%tempy1[,k]
                     +(Phi[,4:5]+AB[,4:5]*tempG)%*%tempy2[,k]
                     +(Phi[,6:7]+AB[,6:7]*tempG)%*%tempy3[,k]+tempvhat
      }  # end loop for k             
      tempy3    <- tempy2
      tempy2    <- tempy1
      tempy1    <- tempy 
      tempst[i] <- mean(tempy[1,])                
    } # end loop i<7
  if(i>7)  {
    for(k in 1:Nb){
      tempvhat  <- rbind(tempvhat1[k,i],tempvhat2[k,i])
      tempG     <- 1/(1+exp(-r*(tempst[i-7]-c)/sdst))#st=y1(t-7)
      tempy[,k] <- Phi[,1]+(Phi[,2:3]+AB[,2:3]*tempG)%*%tempy1[,k]
                   +(Phi[,4:5]+AB[,4:5]*tempG)%*%tempy2[,k]
                   +(Phi[,6:7]+AB[,6:7]*tempG)%*%tempy3[,k]+tempvhat
    } # end loop for k              
    tempy3 <- tempy2
    tempy2 <- tempy1
    tempy1 <- tempy 
  }  # end loop for i>7 
  oneT2newy[,i] <- matrix(c(mean(tempy[1,]),mean(tempy[2,])),2,1)
 } # end loop for i in 1:j
}  # end loop h>1 and h>j
return(oneT2newy)
}

myforecast <- function(data,h=12,H1,H2,PIN=1000){
T      <- length(data[1,])
newy   <- matrix(0,2,h*(H1-H2+1)) 
forePI <- matrix(0,PIN*2,h*(H1-H2+1))  
for(j in H1:H2){
  print(j)
# Add one data once. Point forecast
 y      <- data[,1:(T-j)]
 Tused  <- T-j
 result <- myestimation(data=y)
 res    <- result$residual
 newy[,((H1-j)*h+1):((H1-j+1)*h)] <- myoneT2forecast(y,result,res,h,j)
 # Prediction interval
 r    <- result$gamma
 c    <- result$c
 Phi  <- result$estPhi
 A    <- result$estA  
 B    <- result$estB  
 AB   <- A%*%B
 sdst <- sd(y[1,1:(Tused-7)])
 for(PIk in 1:PIN){
                # For bootstrap data to estimate one set of parameter 
                preInd <- sample(seq(1,(Tused-7),1),Tused-7,replace=TRUE)
                preres <- result$residual[,preInd]
                dataPI <- matrix(0,2,Tused)
                intInd <- sample(seq(1,(Tused-7),1),1,replace=TRUE) 
                dataPI[,1:7] <- y[,intInd:(intInd+6)]
                for(PIt in 8:Tused) {
                  tempG <- 1/(1+exp(-r*(dataPI[1,(PIt-7)]-c)/sdst))  
dataPI[,PIt] <- Phi[,1]+(Phi[,2:3]+AB[,2:3]*tempG)%*%dataPI[,(PIt-1)]+(Phi[,4:5]
+AB[,4:5]*tempG)%*%y[,(PIt-2)]+(Phi[,6:7]+AB[,6:7]*tempG)%*%y[,(PIt-3)]+preres[,PIt-7]
                } # end loop for PIt      
                newresult <- myestimation(dataPI)
                # For one set of errors
                preInd2 <- sample(seq(1,(Tused-7),1),Tused-7,replace=TRUE)
                #newres <- result$residual[,preInd2]
forePI[((PIk-1)*2+1):(PIk*2),((H1-j)*h+1):((H1-j+1)*h)]<-myoneT2forecast(y,
                                                       newresult,res,h,j)
        } # end loop for PIk      
}         # end loop for j in H1:H2   
return(list(pointf=newy,intervalf=forePI))
}
